/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import java.util.Arrays;
import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.exception.util.LocalizedFormats;
import org.apache.commons.math.linear.ArrayRealVector;
import org.apache.commons.math.linear.BlockRealMatrix;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.QRDecomposition;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.util.FastMath;

public class QRDecompositionImpl
implements QRDecomposition {
    private double[][] qrt;
    private double[] rDiag;
    private RealMatrix cachedQ;
    private RealMatrix cachedQT;
    private RealMatrix cachedR;
    private RealMatrix cachedH;

    public QRDecompositionImpl(RealMatrix matrix) {
        int m = matrix.getRowDimension();
        int n = matrix.getColumnDimension();
        this.qrt = matrix.transpose().getData();
        this.rDiag = new double[FastMath.min(m, n)];
        this.cachedQ = null;
        this.cachedQT = null;
        this.cachedR = null;
        this.cachedH = null;
        int minor = 0;
        while (minor < FastMath.min(m, n)) {
            double a;
            double[] qrtMinor = this.qrt[minor];
            double xNormSqr = 0.0;
            int row = minor;
            while (row < m) {
                double c = qrtMinor[row];
                xNormSqr += c * c;
                ++row;
            }
            this.rDiag[minor] = a = qrtMinor[minor] > 0.0 ? -FastMath.sqrt(xNormSqr) : FastMath.sqrt(xNormSqr);
            if (a != 0.0) {
                int n2 = minor;
                qrtMinor[n2] = qrtMinor[n2] - a;
                int col = minor + 1;
                while (col < n) {
                    double[] qrtCol = this.qrt[col];
                    double alpha = 0.0;
                    int row2 = minor;
                    while (row2 < m) {
                        alpha -= qrtCol[row2] * qrtMinor[row2];
                        ++row2;
                    }
                    alpha /= a * qrtMinor[minor];
                    row2 = minor;
                    while (row2 < m) {
                        int n3 = row2;
                        qrtCol[n3] = qrtCol[n3] - alpha * qrtMinor[row2];
                        ++row2;
                    }
                    ++col;
                }
            }
            ++minor;
        }
    }

    @Override
    public RealMatrix getR() {
        if (this.cachedR == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedR = MatrixUtils.createRealMatrix(m, n);
            int row = FastMath.min(m, n) - 1;
            while (row >= 0) {
                this.cachedR.setEntry(row, row, this.rDiag[row]);
                int col = row + 1;
                while (col < n) {
                    this.cachedR.setEntry(row, col, this.qrt[col][row]);
                    ++col;
                }
                --row;
            }
        }
        return this.cachedR;
    }

    @Override
    public RealMatrix getQ() {
        if (this.cachedQ == null) {
            this.cachedQ = this.getQT().transpose();
        }
        return this.cachedQ;
    }

    @Override
    public RealMatrix getQT() {
        if (this.cachedQT == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedQT = MatrixUtils.createRealMatrix(m, m);
            int minor = m - 1;
            while (minor >= FastMath.min(m, n)) {
                this.cachedQT.setEntry(minor, minor, 1.0);
                --minor;
            }
            minor = FastMath.min(m, n) - 1;
            while (minor >= 0) {
                double[] qrtMinor = this.qrt[minor];
                this.cachedQT.setEntry(minor, minor, 1.0);
                if (qrtMinor[minor] != 0.0) {
                    int col = minor;
                    while (col < m) {
                        double alpha = 0.0;
                        int row = minor;
                        while (row < m) {
                            alpha -= this.cachedQT.getEntry(col, row) * qrtMinor[row];
                            ++row;
                        }
                        alpha /= this.rDiag[minor] * qrtMinor[minor];
                        row = minor;
                        while (row < m) {
                            this.cachedQT.addToEntry(col, row, -alpha * qrtMinor[row]);
                            ++row;
                        }
                        ++col;
                    }
                }
                --minor;
            }
        }
        return this.cachedQT;
    }

    @Override
    public RealMatrix getH() {
        if (this.cachedH == null) {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            this.cachedH = MatrixUtils.createRealMatrix(m, n);
            int i2 = 0;
            while (i2 < m) {
                int j = 0;
                while (j < FastMath.min(i2 + 1, n)) {
                    this.cachedH.setEntry(i2, j, this.qrt[j][i2] / -this.rDiag[j]);
                    ++j;
                }
                ++i2;
            }
        }
        return this.cachedH;
    }

    @Override
    public DecompositionSolver getSolver() {
        return new Solver(this.qrt, this.rDiag);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] qrt;
        private final double[] rDiag;

        private Solver(double[][] qrt, double[] rDiag) {
            this.qrt = qrt;
            this.rDiag = rDiag;
        }

        @Override
        public boolean isNonSingular() {
            double[] dArray = this.rDiag;
            int n = this.rDiag.length;
            int n2 = 0;
            while (n2 < n) {
                double diag = dArray[n2];
                if (diag == 0.0) {
                    return false;
                }
                ++n2;
            }
            return true;
        }

        @Override
        public double[] solve(double[] b) throws IllegalArgumentException, InvalidMatrixException {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b.length != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b.length, m);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            double[] x = new double[n];
            double[] y = (double[])b.clone();
            int minor = 0;
            while (minor < FastMath.min(m, n)) {
                double[] qrtMinor = this.qrt[minor];
                double dotProduct = 0.0;
                int row = minor;
                while (row < m) {
                    dotProduct += y[row] * qrtMinor[row];
                    ++row;
                }
                dotProduct /= this.rDiag[minor] * qrtMinor[minor];
                row = minor;
                while (row < m) {
                    int n2 = row;
                    y[n2] = y[n2] + dotProduct * qrtMinor[row];
                    ++row;
                }
                ++minor;
            }
            int row = this.rDiag.length - 1;
            while (row >= 0) {
                int n3 = row;
                y[n3] = y[n3] / this.rDiag[row];
                double yRow = y[row];
                double[] qrtRow = this.qrt[row];
                x[row] = yRow;
                int i2 = 0;
                while (i2 < row) {
                    int n4 = i2;
                    y[n4] = y[n4] - yRow * qrtRow[i2];
                    ++i2;
                }
                --row;
            }
            return x;
        }

        @Override
        public RealVector solve(RealVector b) throws IllegalArgumentException, InvalidMatrixException {
            try {
                return this.solve((ArrayRealVector)b);
            }
            catch (ClassCastException cce) {
                return new ArrayRealVector(this.solve(b.getData()), false);
            }
        }

        public ArrayRealVector solve(ArrayRealVector b) throws IllegalArgumentException, InvalidMatrixException {
            return new ArrayRealVector(this.solve(b.getDataRef()), false);
        }

        @Override
        public RealMatrix solve(RealMatrix b) throws IllegalArgumentException, InvalidMatrixException {
            int n = this.qrt.length;
            int m = this.qrt[0].length;
            if (b.getRowDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.DIMENSIONS_MISMATCH_2x2, b.getRowDimension(), b.getColumnDimension(), m, "n");
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int columns = b.getColumnDimension();
            int blockSize = 52;
            int cBlocks = (columns + 52 - 1) / 52;
            double[][] xBlocks = BlockRealMatrix.createBlocksLayout(n, columns);
            double[][] y = new double[b.getRowDimension()][52];
            double[] alpha = new double[52];
            int kBlock = 0;
            while (kBlock < cBlocks) {
                int kStart = kBlock * 52;
                int kEnd = FastMath.min(kStart + 52, columns);
                int kWidth = kEnd - kStart;
                b.copySubMatrix(0, m - 1, kStart, kEnd - 1, y);
                int minor = 0;
                while (minor < FastMath.min(m, n)) {
                    int k;
                    double[] yRow;
                    double d;
                    double[] qrtMinor = this.qrt[minor];
                    double factor = 1.0 / (this.rDiag[minor] * qrtMinor[minor]);
                    Arrays.fill(alpha, 0, kWidth, 0.0);
                    int row = minor;
                    while (row < m) {
                        d = qrtMinor[row];
                        yRow = y[row];
                        k = 0;
                        while (k < kWidth) {
                            int n2 = k;
                            alpha[n2] = alpha[n2] + d * yRow[k];
                            ++k;
                        }
                        ++row;
                    }
                    int k2 = 0;
                    while (k2 < kWidth) {
                        int n3 = k2++;
                        alpha[n3] = alpha[n3] * factor;
                    }
                    row = minor;
                    while (row < m) {
                        d = qrtMinor[row];
                        yRow = y[row];
                        k = 0;
                        while (k < kWidth) {
                            int n4 = k;
                            yRow[n4] = yRow[n4] + alpha[k] * d;
                            ++k;
                        }
                        ++row;
                    }
                    ++minor;
                }
                int j = this.rDiag.length - 1;
                while (j >= 0) {
                    int jBlock = j / 52;
                    int jStart = jBlock * 52;
                    double factor = 1.0 / this.rDiag[j];
                    double[] yJ = y[j];
                    double[] xBlock = xBlocks[jBlock * cBlocks + kBlock];
                    int index = (j - jStart) * kWidth;
                    int k = 0;
                    while (k < kWidth) {
                        int n5 = k;
                        yJ[n5] = yJ[n5] * factor;
                        xBlock[index++] = yJ[k];
                        ++k;
                    }
                    double[] qrtJ = this.qrt[j];
                    int i2 = 0;
                    while (i2 < j) {
                        double rIJ = qrtJ[i2];
                        double[] yI = y[i2];
                        int k3 = 0;
                        while (k3 < kWidth) {
                            int n6 = k3;
                            yI[n6] = yI[n6] - yJ[k3] * rIJ;
                            ++k3;
                        }
                        ++i2;
                    }
                    --j;
                }
                ++kBlock;
            }
            return new BlockRealMatrix(n, columns, xBlocks, false);
        }

        @Override
        public RealMatrix getInverse() throws InvalidMatrixException {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.rDiag.length));
        }
    }
}

